/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include <cstddef>
#include <cstdint>
#include <memory>
#include <msgpack.hpp>
#include <optional>

enum class PacketType
{
	Request = 0,
	Response = 1,
	Notification = 2,
	Error = 1 << 7,
	ErrorResponse = (1 << 7) | Response,
	ErrorNotification = (1 << 7) | Notification,
	Invalid = 255,
};

#define _MessageLenHeader 4
#define _HeaderLen 2
#define _FullHeaderLen (_HeaderLen + _MessageLenHeader)
class Packet
{
private:
	PacketType _type;

protected:
	static const std::uint8_t PacketStart;
	static const std::uint8_t MessageLen[];
	msgpack::sbuffer Stream;

public:
	static const std::int32_t FullHeaderLen;
	static const std::int32_t HeaderLen;
	std::optional<std::uint32_t> MessageId;

	std::size_t Size() const noexcept { return Stream.size(); }

	bool IsRequest() const noexcept { return _type == PacketType::Request; }
	bool IsResponse() const noexcept { return _type == PacketType::Response; }
	bool IsNotification() const noexcept { return ((int)_type & (int)PacketType::Notification) != 0; }
	bool IsError() const noexcept { return ((int)_type & (int)PacketType::Error) != 0; }
	PacketType GetType() const noexcept { return _type; }

	static void GetHeader(PacketType type, std::uint8_t* header)
	{
		header[0] = PacketStart;
		header[1] = (std::uint8_t)type;
	}

	Packet(const char* uint8_ts, std::uint32_t len, PacketType type, std::optional<std::uint32_t> id = {})
	    : _type(type),
	      Stream(),
	      MessageId(id)
	{
		Stream.write(uint8_ts, len);
	}

	virtual ~Packet() {}
};
